"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Plus, Search } from "lucide-react"
import { useSearchParams, useRouter, usePathname } from "next/navigation"

interface Appointment {
    id: string
    appointment_date: string
    appointment_time: string
    status: string
    services: { name: string } | null
    profiles: { full_name: string } | null
    branches: { name: string } | null
}

interface Branch {
    id: string
    name: string
}

interface Service {
    id: string
    name: string
    price: number
}

interface Profile {
    id: string
    full_name: string
    email: string
}

export function AppointmentsView() {
    const router = useRouter()
    const pathname = usePathname()
    const searchParams = useSearchParams()

    const [appointments, setAppointments] = useState<Appointment[]>([])
    const [branches, setBranches] = useState<Branch[]>([])
    const [services, setServices] = useState<Service[]>([])
    const [profiles, setProfiles] = useState<Profile[]>([])

    const [isCreateModalOpen, setIsCreateModalOpen] = useState(false)

    const [newAppointment, setNewAppointment] = useState({
        profileId: "",
        newCustomerName: "",
        newCustomerEmail: "",
        serviceId: "",
        branchId: "",
        date: "",
        time: "",
        isNewCustomer: false
    })

    // Read params
    const selectedDate = searchParams.get("date") || ""
    const selectedBranch = searchParams.get("branch") || "all"
    const searchQuery = searchParams.get("search") || ""

    // Update params helper
    const updateParams = (key: string, value: string) => {
        const params = new URLSearchParams(searchParams.toString())
        if (value && value !== "all") {
            params.set(key, value)
        } else {
            params.delete(key)
        }
        router.replace(`${pathname}?${params.toString()}`)
    }

    useEffect(() => {
        fetchInitialData()
    }, [])

    useEffect(() => {
        fetchAppointments()
    }, [selectedDate, selectedBranch, searchQuery])

    const fetchInitialData = async () => {
        const supabase = createClient()
        const { data: branchesData } = await supabase.from("branches").select("id, name").order("name")
        setBranches(branchesData || [])

        const { data: servicesData } = await supabase.from("services").select("id, name, price").order("name")
        setServices(servicesData || [])

        const { data: profilesData } = await supabase.from("profiles").select("id, full_name, email").order("full_name")
        setProfiles(profilesData || [])
    }

    const fetchAppointments = async () => {
        const supabase = createClient()
        let query = supabase
            .from("appointments")
            .select("id, appointment_date, appointment_time, status, services(name), profiles(full_name), branches(name)")
            .order("appointment_date", { ascending: true })

        if (selectedDate) {
            query = query.eq("appointment_date", selectedDate)
        }

        if (selectedBranch && selectedBranch !== "all") {
            query = query.eq("branch_id", selectedBranch)
        }

        const { data } = await query

        let filteredData = (data as unknown as Appointment[]) || []

        if (searchQuery) {
            const lowerQuery = searchQuery.toLowerCase()
            filteredData = filteredData.filter(app =>
                app.profiles?.full_name?.toLowerCase().includes(lowerQuery) ||
                app.services?.name?.toLowerCase().includes(lowerQuery)
            )
        }

        setAppointments(filteredData)
    }

    const handleCreateAppointment = async (e: React.FormEvent) => {
        e.preventDefault()
        const supabase = createClient()

        try {
            let userId = newAppointment.profileId

            if (newAppointment.isNewCustomer) {
                alert("Creating new users from admin panel requires backend function. Please select distinct user.")
                return
            }

            const { error } = await supabase.from("appointments").insert({
                user_id: userId,
                service_id: newAppointment.serviceId,
                branch_id: newAppointment.branchId,
                appointment_date: newAppointment.date,
                appointment_time: newAppointment.time,
                status: 'confirmed'
            })

            if (error) throw error

            setIsCreateModalOpen(false)
            fetchAppointments()
            setNewAppointment({
                profileId: "",
                newCustomerName: "",
                newCustomerEmail: "",
                serviceId: "",
                branchId: "",
                date: "",
                time: "",
                isNewCustomer: false
            })

        } catch (error) {
            console.error(error)
            alert("Failed to create appointment")
        }
    }

    return (
        <div className="p-6 space-y-6">
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                <h2 className="text-2xl font-bold text-foreground">Appointments</h2>
                <Dialog open={isCreateModalOpen} onOpenChange={setIsCreateModalOpen}>
                    <DialogTrigger asChild>
                        <Button className="bg-gold hover:bg-gold-dark text-black">
                            <Plus className="w-4 h-4 mr-2" />
                            New Appointment
                        </Button>
                    </DialogTrigger>
                    <DialogContent className="max-w-md bg-card border-border">
                        <DialogHeader>
                            <DialogTitle>Create Appointment</DialogTitle>
                        </DialogHeader>
                        <form onSubmit={handleCreateAppointment} className="space-y-4 py-4">
                            <div className="space-y-2">
                                <Label>Customer</Label>
                                <Select
                                    value={newAppointment.profileId}
                                    onValueChange={(val) => setNewAppointment({ ...newAppointment, profileId: val })}
                                >
                                    <SelectTrigger className="bg-input border-border">
                                        <SelectValue placeholder="Select Customer" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        {profiles.map(profile => (
                                            <SelectItem key={profile.id} value={profile.id}>{profile.full_name} ({profile.email})</SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="space-y-2">
                                <Label>Service</Label>
                                <Select
                                    value={newAppointment.serviceId}
                                    onValueChange={(val) => setNewAppointment({ ...newAppointment, serviceId: val })}
                                >
                                    <SelectTrigger className="bg-input border-border">
                                        <SelectValue placeholder="Select Service" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        {services.map(service => (
                                            <SelectItem key={service.id} value={service.id}>{service.name} (${service.price})</SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="space-y-2">
                                <Label>Branch</Label>
                                <Select
                                    value={newAppointment.branchId}
                                    onValueChange={(val) => setNewAppointment({ ...newAppointment, branchId: val })}
                                >
                                    <SelectTrigger className="bg-input border-border">
                                        <SelectValue placeholder="Select Branch" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        {branches.map(branch => (
                                            <SelectItem key={branch.id} value={branch.id}>{branch.name}</SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label>Date</Label>
                                    <Input
                                        type="date"
                                        value={newAppointment.date}
                                        onChange={(e) => setNewAppointment({ ...newAppointment, date: e.target.value })}
                                        className="bg-input border-border"
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Time</Label>
                                    <Input
                                        type="time"
                                        value={newAppointment.time}
                                        onChange={(e) => setNewAppointment({ ...newAppointment, time: e.target.value })}
                                        className="bg-input border-border"
                                        required
                                    />
                                </div>
                            </div>

                            <Button type="submit" className="w-full bg-gold hover:bg-gold-dark text-black mt-4">
                                Book Appointment
                            </Button>
                        </form>
                    </DialogContent>
                </Dialog>
            </div>

            <div className="flex flex-col md:flex-row gap-4 bg-card p-4 rounded-lg border border-border">
                <div className="w-full md:w-1/3 relative">
                    <Search className="absolute left-3 top-2.5 h-4 w-4 text-muted-foreground" />
                    <Input
                        placeholder="Search customer..."
                        value={searchQuery}
                        onChange={(e) => updateParams("search", e.target.value)}
                        className="pl-9 bg-background border-border"
                    />
                </div>
                <div className="w-full md:w-1/4">
                    <Input
                        type="date"
                        value={selectedDate}
                        onChange={(e) => updateParams("date", e.target.value)}
                        className="bg-background border-border"
                    />
                </div>
                <div className="w-full md:w-1/4">
                    <select
                        value={selectedBranch}
                        onChange={(e) => updateParams("branch", e.target.value)}
                        className="w-full h-10 px-3 py-2 bg-background border border-border rounded-lg text-foreground focus:outline-none focus:ring-2 focus:ring-ring focus:ring-offset-2"
                    >
                        <option value="all">All Branches</option>
                        {branches.map(branch => (
                            <option key={branch.id} value={branch.id}>{branch.name}</option>
                        ))}
                    </select>
                </div>
            </div>

            <div className="bg-card border border-border rounded-lg overflow-hidden">
                <div className="overflow-x-auto">
                    <table className="w-full text-sm text-left">
                        <thead className="bg-secondary text-foreground uppercase border-b border-border">
                            <tr>
                                <th className="px-6 py-3 font-semibold">Client</th>
                                <th className="px-6 py-3 font-semibold">Service</th>
                                <th className="px-6 py-3 font-semibold">Date & Time</th>
                                <th className="px-6 py-3 font-semibold">Branch</th>
                                <th className="px-6 py-3 font-semibold">Status</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-border">
                            {appointments?.length > 0 ? (
                                appointments.map((app) => (
                                    <tr key={app.id} className="hover:bg-muted/50 transition-colors">
                                        <td className="px-6 py-4 font-medium text-foreground">
                                            {app.profiles?.full_name || "Unknown"}
                                        </td>
                                        <td className="px-6 py-4 text-muted-foreground">
                                            {app.services?.name}
                                        </td>
                                        <td className="px-6 py-4 text-muted-foreground">
                                            {new Date(app.appointment_date).toLocaleDateString()} at {app.appointment_time}
                                        </td>
                                        <td className="px-6 py-4 text-muted-foreground">
                                            {app.branches?.name}
                                        </td>
                                        <td className="px-6 py-4">
                                            <span className={`px-2 py-1 rounded-full text-xs font-semibold
                                                ${app.status === 'confirmed' ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' :
                                                    app.status === 'cancelled' ? 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400' :
                                                        'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400'}`}>
                                                {app.status}
                                            </span>
                                        </td>
                                    </tr>
                                ))
                            ) : (
                                <tr>
                                    <td colSpan={5} className="px-6 py-8 text-center text-muted-foreground">
                                        No appointments found.
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    )
}
