"use client"

import { useState } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Loader2 } from "lucide-react"

interface Branch {
    id: string
    name: string
    city: string
}

interface GalleryUploadModalProps {
    isOpen: boolean
    onOpenChange: (open: boolean) => void
    branches: Branch[]
    onUploadComplete: (newImages: any[]) => void
}

export function GalleryUploadModal({ isOpen, onOpenChange, branches, onUploadComplete }: GalleryUploadModalProps) {
    const [isUploading, setIsUploading] = useState(false)
    const [newImage, setNewImage] = useState({
        title: "",
        category: "",
        branchId: "all",
        files: [] as File[]
    })

    const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files && e.target.files.length > 0) {
            setNewImage({ ...newImage, files: Array.from(e.target.files) })
        }
    }

    const handleUpload = async (e: React.FormEvent) => {
        e.preventDefault()
        if (newImage.files.length === 0) return

        setIsUploading(true)
        const supabase = createClient()

        try {
            const uploadPromises = newImage.files.map(async (file) => {
                const fileExt = file.name.split('.').pop()
                const fileName = `${Date.now()}-${Math.random().toString(36).substring(7)}.${fileExt}`
                const filePath = `gallery/${fileName}`

                const { error: uploadError } = await supabase.storage
                    .from('united-nails-uploads')
                    .upload(filePath, file, {
                        metadata: {
                            branchId: newImage.branchId === "all" ? null : newImage.branchId
                        }
                    })

                if (uploadError) throw uploadError

                const { data: { publicUrl } } = supabase.storage
                    .from('united-nails-uploads')
                    .getPublicUrl(filePath)

                return {
                    title: newImage.title || file.name.split('.').slice(0, -1).join('.'),
                    category: newImage.category || null,
                    image_url: publicUrl,
                    branch_id: newImage.branchId === "all" ? null : newImage.branchId
                }
            })

            const imagesToInsert = await Promise.all(uploadPromises)

            const { data: dbData, error: dbError } = await supabase
                .from("gallery_images")
                .insert(imagesToInsert)
                .select("*, branches(name, city)")

            if (dbError) throw dbError

            if (dbData) {
                onUploadComplete(dbData)
            }

            onOpenChange(false)
            setNewImage({ title: "", category: "", branchId: "all", files: [] })

        } catch (error) {
            console.error(error)
            alert("Failed to upload images")
        } finally {
            setIsUploading(false)
        }
    }

    return (
        <Dialog open={isOpen} onOpenChange={onOpenChange}>
            <DialogContent className="bg-card border-border">
                <DialogHeader>
                    <DialogTitle>Upload Gallery Images</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleUpload} className="space-y-4 py-4">
                    <div>
                        <Label>Title (Optional)</Label>
                        <Input
                            value={newImage.title}
                            onChange={(e) => setNewImage({ ...newImage, title: e.target.value })}
                            className="mt-2 bg-input border-border"
                            placeholder="Leave empty to use filename"
                        />
                    </div>

                    <div>
                        <Label>Category</Label>
                        <select
                            value={newImage.category}
                            onChange={(e) => setNewImage({ ...newImage, category: e.target.value })}
                            className="w-full mt-2 px-3 py-2 bg-input border border-border rounded-lg text-foreground"
                        >
                            <option value="">All Category</option>
                            <option>Nail Art</option>
                            <option>Manicure</option>
                            <option>Pedicure</option>
                            <option>Salon</option>
                        </select>
                    </div>

                    <div>
                        <Label>Branch (Optional)</Label>
                        <select
                            value={newImage.branchId}
                            onChange={(e) => setNewImage({ ...newImage, branchId: e.target.value })}
                            className="w-full mt-2 px-3 py-2 bg-input border border-border rounded-lg text-foreground"
                        >
                            <option value="all">Global / All Branches</option>
                            {branches.map(branch => (
                                <option key={branch.id} value={branch.id}>{branch.name} ({branch.city})</option>
                            ))}
                        </select>
                    </div>

                    <div>
                        <Label>Images</Label>
                        <Input
                            type="file"
                            accept="image/*"
                            multiple
                            onChange={handleFileChange}
                            className="mt-2 bg-input border-border"
                            required
                        />
                        <p className="text-xs text-muted-foreground mt-1">
                            {newImage.files.length > 0 ? `${newImage.files.length} files selected` : "Select multiple files"}
                        </p>
                    </div>

                    <Button type="submit" disabled={isUploading} className="w-full bg-gold hover:bg-gold-dark text-black mt-4">
                        {isUploading ? <><Loader2 className="w-4 h-4 mr-2 animate-spin" /> Uploading...</> : "Upload Images"}
                    </Button>
                </form>
            </DialogContent>
        </Dialog>
    )
}
