"use client"

import { useState, useEffect } from "react"
import { useParams, useRouter } from "next/navigation"
import { createClient } from "@/lib/supabase/client"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { ArrowLeft, Plus, Pencil, Trash2 } from "lucide-react"
import { ConfirmationModal } from "./confirmation-modal"
import { ServiceModal } from "./service-modal"
import { currencyFormat } from "@/utils/common"

interface Service {
    id: string
    name: string
    description: string | null
    sub_category_id: string
    created_at: string
    service_prices?: ServicePrice[]
}

interface ServicePrice {
    id: string
    service_id: string
    branch_id: string
    price: number
    branches?: { name: string; city: string }
}

interface Subcategory {
    id: string
    name: string
}

export function ServicesManagementView() {
    const params = useParams()
    const router = useRouter()
    const categoryId = params.categoryId as string
    const subcategoryId = params.subcategoryId as string

    const [services, setServices] = useState<Service[]>([])
    const [subcategory, setSubcategory] = useState<Subcategory | null>(null)
    const [isModalOpen, setIsModalOpen] = useState(false)
    const [editingService, setEditingService] = useState<Service | null>(null)
    const [deletingService, setDeletingService] = useState<Service | null>(null)
    const [isDeleteLoading, setIsDeleteLoading] = useState(false)

    useEffect(() => {
        fetchSubcategory()
        fetchServices()
    }, [subcategoryId])

    const fetchSubcategory = async () => {
        const supabase = createClient()
        const { data } = await supabase
            .from("sub_categories")
            .select("id, name")
            .eq("id", subcategoryId)
            .single()

        setSubcategory(data)
    }

    const fetchServices = async () => {
        try {
            const supabase = createClient()

            const { data: servicesData, error: servicesError } = await supabase
                .from("services")
                .select("*")
                .eq("sub_category_id", subcategoryId)
                .order("created_at")

            if (servicesError) {
                console.error("Error fetching services:", servicesError)
                setServices([])
                return
            }

            if (!servicesData || servicesData.length === 0) {
                setServices([])
                return
            }

            const serviceIds = servicesData.map(s => s.id)

            const { data: pricesData, error: pricesError } = await supabase
                .from("service_prices")
                .select(`
                    id,
                    service_id,
                    branch_id,
                    price,
                    branches(name, city)
                `)
                .in("service_id", serviceIds)

            if (pricesError) {
                console.error("Error fetching prices:", pricesError)
            }

            const servicesWithPrices = servicesData.map(service => ({
                ...service,
                service_prices: pricesData?.filter(p => p.service_id === service.id) || []
            }))

            setServices(servicesWithPrices)
        } catch (error) {
            console.error("Exception in fetchServices:", error)
            setServices([])
        }
    }

    const openCreateModal = () => {
        setEditingService(null)
        setIsModalOpen(true)
    }

    const handleEdit = (service: Service, e: React.MouseEvent) => {
        e.stopPropagation()
        setEditingService(service)
        setIsModalOpen(true)
    }

    const handleDeleteClick = (service: Service, e: React.MouseEvent) => {
        e.stopPropagation()
        setDeletingService(service)
    }

    const confirmDelete = async () => {
        if (!deletingService) return
        setIsDeleteLoading(true)
        const supabase = createClient()
        try {
            const { error } = await supabase
                .from("services")
                .delete()
                .eq("id", deletingService.id)
            if (error) throw error
            await fetchServices()
            setDeletingService(null)
        } catch (error) {
            console.error(error)
            alert("Failed to delete service")
        } finally {
            setIsDeleteLoading(false)
        }
    }

    return (
        <div className="p-6 space-y-6">
            <div className="flex items-center gap-4">
                <Button
                    variant="ghost"
                    size="icon"
                    onClick={() => router.push(`/admin/services/${categoryId}`)}
                    className="hover:bg-secondary"
                >
                    <ArrowLeft className="w-5 h-5" />
                </Button>
                <h2 className="text-2xl font-bold text-foreground">
                    {subcategory?.name}
                </h2>
            </div>

            <div className="flex items-center justify-between">
                <p className="text-muted-foreground">
                    Manage services and pricing for {subcategory?.name}
                </p>
                <Button
                    onClick={openCreateModal}
                    className="bg-gold hover:bg-gold-dark text-black gap-2"
                >
                    Add Service <Plus className="w-4 h-4" />
                </Button>
            </div>

            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                {services.map((service) => (
                    <Card key={service.id} className="border-border hover:border-gold transition-colors group relative cursor-pointer">
                        <CardContent className="p-6">
                            <div className="flex items-start justify-between mb-3">
                                <div className="flex-1">
                                    <h3 className="font-semibold text-lg text-foreground">{service.name}</h3>
                                    {service.description && (
                                        <p className="text-sm text-muted-foreground mt-1">{service.description}</p>
                                    )}
                                </div>
                                <div className="flex gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                    <Button
                                        variant="ghost"
                                        size="icon"
                                        onClick={(e) => handleEdit(service, e)}
                                        className="h-8 w-8 hover:text-gold"
                                    >
                                        <Pencil className="w-4 h-4" />
                                    </Button>
                                    <Button
                                        variant="ghost"
                                        size="icon"
                                        onClick={(e) => handleDeleteClick(service, e)}
                                        className="h-8 w-8 hover:text-destructive"
                                    >
                                        <Trash2 className="w-4 h-4" />
                                    </Button>
                                </div>
                            </div>
                            {service.service_prices && service.service_prices.length > 0 && (
                                <div className="mt-3 pt-3 border-t border-border space-y-2">
                                    {service.service_prices.map((price) => (
                                        <div key={price.id} className="flex justify-between items-center text-sm">
                                            <span className="text-muted-foreground">
                                                {price.branches?.name} - {price.branches?.city}
                                            </span>
                                            <span className="font-semibold text-gold">
                                                {currencyFormat(price.price)}
                                            </span>
                                        </div>
                                    ))}
                                </div>
                            )}
                        </CardContent>
                    </Card>
                ))}
            </div>

            <ServiceModal
                isOpen={isModalOpen}
                onOpenChange={setIsModalOpen}
                onSuccess={fetchServices}
                subcategoryId={subcategoryId}
                service={editingService}
            />

            <ConfirmationModal
                isOpen={!!deletingService}
                onOpenChange={(open) => !open && setDeletingService(null)}
                title="Delete Service"
                description={`Are you sure you want to delete "${deletingService?.name}"? This will also delete all associated pricing. This action cannot be undone.`}
                onConfirm={confirmDelete}
                isLoading={isDeleteLoading}
                variant="destructive"
                confirmLabel="Delete"
            />
        </div>
    )
}
