"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Loader2 } from "lucide-react"

interface Subcategory {
    id: string
    name: string
    category_id: string
    created_at?: string
}

interface SubcategoryModalProps {
    isOpen: boolean
    onOpenChange: (open: boolean) => void
    onSuccess: () => void
    categoryId: string
    subcategory?: Subcategory | null
}

export function SubcategoryModal({ isOpen, onOpenChange, onSuccess, categoryId, subcategory }: SubcategoryModalProps) {
    const [isLoading, setIsLoading] = useState(false)
    const [name, setName] = useState("")

    useEffect(() => {
        if (subcategory) {
            setName(subcategory.name)
        } else {
            setName("")
        }
    }, [subcategory, isOpen])

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault()
        setIsLoading(true)
        const supabase = createClient()

        try {
            if (subcategory) {
                const { error } = await supabase
                    .from("sub_categories")
                    .update({ name })
                    .eq("id", subcategory.id)
                if (error) throw error
            } else {
                const { error } = await supabase
                    .from("sub_categories")
                    .insert([{ name, category_id: categoryId }])
                if (error) throw error
            }
            onSuccess()
            onOpenChange(false)
            setName("")
        } catch (error) {
            console.error(error)
            alert(subcategory ? "Failed to update subcategory" : "Failed to create subcategory")
        } finally {
            setIsLoading(false)
        }
    }

    return (
        <Dialog open={isOpen} onOpenChange={onOpenChange}>
            <DialogContent className="bg-card border-border">
                <DialogHeader>
                    <DialogTitle>{subcategory ? "Edit Subcategory" : "Add New Subcategory"}</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4 py-4">
                    <div>
                        <Label>Subcategory Name</Label>
                        <Input
                            value={name}
                            onChange={(e) => setName(e.target.value)}
                            className="mt-2"
                            required
                        />
                    </div>
                    <Button type="submit" disabled={isLoading} className="w-full bg-gold hover:bg-gold-dark text-black mt-4">
                        {isLoading ? <><Loader2 className="w-4 h-4 mr-2 animate-spin" /> {subcategory ? "Updating..." : "Creating..."}</> : (subcategory ? "Update Subcategory" : "Create Subcategory")}
                    </Button>
                </form>
            </DialogContent>
        </Dialog>
    )
}
