"use client"

import { useState, useEffect } from "react"
import { createClient } from "@/lib/supabase/client"
import { Button } from "@/components/ui/button"
import { Plus, Pencil, Trash2, Star } from "lucide-react"
import { ConfirmationModal } from "./confirmation-modal"
import { TestimonialModal } from "./testimonial-modal"

interface Testimonial {
    id: string
    user_name: string
    user_image: string | null
    rating: number
    message: string
    branch_id: string | null
    branches?: { name: string; city: string } | null
    created_at: string
}

export function TestimonialsView() {
    const [testimonials, setTestimonials] = useState<Testimonial[]>([])
    const [isModalOpen, setIsModalOpen] = useState(false)
    const [editingTestimonial, setEditingTestimonial] = useState<Testimonial | null>(null)
    const [deletingTestimonial, setDeletingTestimonial] = useState<Testimonial | null>(null)
    const [isDeleteLoading, setIsDeleteLoading] = useState(false)

    useEffect(() => {
        fetchTestimonials()
    }, [])

    const fetchTestimonials = async () => {
        try {
            const supabase = createClient()
            const { data: testimonialsData, error: testimonialsError } = await supabase
                .from("testimonials")
                .select("*")
                .order("created_at", { ascending: false })
                .limit(20)

            if (testimonialsError) {
                console.error("Error fetching testimonials:", testimonialsError)
                setTestimonials([])
                return
            }

            if (!testimonialsData || testimonialsData.length === 0) {
                setTestimonials([])
                return
            }

            const branchIds = testimonialsData
                .map(t => t.branch_id)
                .filter(id => id !== null)

            if (branchIds.length > 0) {
                const { data: branchesData } = await supabase
                    .from("branches")
                    .select("id, name, city")
                    .in("id", branchIds)

                const testimonialsWithBranches = testimonialsData.map(testimonial => ({
                    ...testimonial,
                    branches: branchesData?.find(b => b.id === testimonial.branch_id) || null
                }))

                setTestimonials(testimonialsWithBranches)
            } else {
                setTestimonials(testimonialsData.map(t => ({ ...t, branches: null })))
            }
        } catch (error) {
            console.error("Exception in fetchTestimonials:", error)
            setTestimonials([])
        }
    }

    const openCreateModal = () => {
        setEditingTestimonial(null)
        setIsModalOpen(true)
    }

    const handleEdit = (testimonial: Testimonial) => {
        setEditingTestimonial(testimonial)
        setIsModalOpen(true)
    }

    const handleDeleteClick = (testimonial: Testimonial) => {
        setDeletingTestimonial(testimonial)
    }

    const confirmDelete = async () => {
        if (!deletingTestimonial) return
        setIsDeleteLoading(true)
        const supabase = createClient()
        try {
            const { error } = await supabase
                .from("testimonials")
                .delete()
                .eq("id", deletingTestimonial.id)
            if (error) throw error
            await fetchTestimonials()
            setDeletingTestimonial(null)
        } catch (error) {
            console.error(error)
            alert("Failed to delete testimonial")
        } finally {
            setIsDeleteLoading(false)
        }
    }

    const renderStars = (rating: number) => {
        return (
            <div className="flex gap-1">
                {[1, 2, 3, 4, 5].map((star) => (
                    <Star
                        key={star}
                        className={`w-4 h-4 ${star <= rating ? "fill-gold text-gold" : "text-muted-foreground"
                            }`}
                    />
                ))}
            </div>
        )
    }

    return (
        <div className="p-6 space-y-6">
            <div className="flex items-center justify-between">
                <h2 className="text-2xl font-bold text-foreground">Testimonials</h2>
                <Button
                    onClick={openCreateModal}
                    className="bg-gold hover:bg-gold-dark text-black gap-2"
                >
                    Add Testimonial <Plus className="w-4 h-4" />
                </Button>
            </div>

            <div className="border border-border rounded-lg overflow-hidden">
                <div className="overflow-x-auto">
                    <table className="w-full">
                        <thead className="bg-secondary">
                            <tr>
                                <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                                    User
                                </th>
                                <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                                    Rating
                                </th>
                                <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                                    Comment
                                </th>
                                <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                                    Branch
                                </th>
                                <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                                    Date
                                </th>
                                <th className="px-6 py-3 text-right text-xs font-medium text-muted-foreground uppercase tracking-wider">
                                    Actions
                                </th>
                            </tr>
                        </thead>
                        <tbody className="bg-card divide-y divide-border">
                            {testimonials.map((testimonial) => (
                                <tr key={testimonial.id} className="hover:bg-secondary/50 transition-colors">
                                    <td className="px-6 py-4 whitespace-nowrap">
                                        <div className="flex items-center gap-3">
                                            {testimonial.user_image ? (
                                                <img
                                                    src={testimonial.user_image}
                                                    alt={testimonial.user_name}
                                                    className="w-10 h-10 rounded-full object-cover"
                                                />
                                            ) : (
                                                <div className="w-10 h-10 rounded-full bg-secondary flex items-center justify-center">
                                                    <span className="text-sm font-medium text-muted-foreground">
                                                        {testimonial.user_name.charAt(0).toUpperCase()}
                                                    </span>
                                                </div>
                                            )}
                                            <span className="font-medium text-foreground">
                                                {testimonial.user_name}
                                            </span>
                                        </div>
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap">
                                        {renderStars(testimonial.rating)}
                                    </td>
                                    <td className="px-6 py-4">
                                        <p className="text-sm text-foreground line-clamp-2 max-w-md">
                                            {testimonial.message}
                                        </p>
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap">
                                        {testimonial.branches ? (
                                            <div className="text-sm">
                                                <div className="font-medium text-foreground">{testimonial.branches.name}</div>
                                                <div className="text-muted-foreground">{testimonial.branches.city}</div>
                                            </div>
                                        ) : (
                                            <span className="text-sm text-muted-foreground">-</span>
                                        )}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-muted-foreground">
                                        {new Date(testimonial.created_at).toLocaleDateString()}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-right">
                                        <div className="flex gap-2 justify-end">
                                            <Button
                                                variant="ghost"
                                                size="icon"
                                                onClick={() => handleEdit(testimonial)}
                                                className="h-8 w-8 hover:text-gold"
                                            >
                                                <Pencil className="w-4 h-4" />
                                            </Button>
                                            <Button
                                                variant="ghost"
                                                size="icon"
                                                onClick={() => handleDeleteClick(testimonial)}
                                                className="h-8 w-8 hover:text-destructive"
                                            >
                                                <Trash2 className="w-4 h-4" />
                                            </Button>
                                        </div>
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>
            </div>

            <TestimonialModal
                isOpen={isModalOpen}
                onOpenChange={setIsModalOpen}
                onSuccess={fetchTestimonials}
                testimonial={editingTestimonial}
            />

            <ConfirmationModal
                isOpen={!!deletingTestimonial}
                onOpenChange={(open) => !open && setDeletingTestimonial(null)}
                title="Delete Testimonial"
                description={`Are you sure you want to delete the testimonial from "${deletingTestimonial?.user_name}"? This action cannot be undone.`}
                onConfirm={confirmDelete}
                isLoading={isDeleteLoading}
                variant="destructive"
                confirmLabel="Delete"
            />
        </div>
    )
}
