"use client"
import { useEffect, useState } from "react"
import { MapPin, Phone, ArrowRight } from "lucide-react"
import Image from "next/image"
import { motion, AnimatePresence } from "framer-motion"
import { useBranches, Branch } from "@/context/branch-context"

function BranchCard({ branch, index }: { branch: Branch; index: number }) {
  const [currentImageIndex, setCurrentImageIndex] = useState(0)

  useEffect(() => {
    if (branch?.images?.length <= 1) return

    const staggerDelay = index * 1000

    let timer: NodeJS.Timeout;

    const timeout = setTimeout(() => {
      timer = setInterval(() => {
        setCurrentImageIndex((prev) => (prev + 1) % branch?.images?.length)
      }, 8000)

    }, staggerDelay)

    return () => {
      clearTimeout(timeout)
      if (timer) {
        clearInterval(timer)
      }
    }
  }, [branch?.images?.length, index])

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      whileInView={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.6, ease: "easeOut" }}
      className="bg-card dark:bg-card rounded-2xl shadow-xl hover:shadow-2xl transition-all duration-500 border border-border overflow-hidden flex flex-col group"
    >
      <div className="relative h-96 w-full overflow-hidden">
        <AnimatePresence mode="wait">
          {branch?.images?.length > 0 ? (
            <motion.div
              key={branch?.images?.[currentImageIndex]}
              initial={{ opacity: 0, scale: 1.1 }}
              animate={{ opacity: 1, scale: 1 }}
              exit={{ opacity: 0, scale: 0.95 }}
              transition={{ duration: 1, ease: "easeInOut" }}
              className="absolute inset-0"
            >
              <Image
                src={branch.images[currentImageIndex]}
                alt={branch.name}
                fill
                className="object-cover transition-transform duration-1000 group-hover:scale-110"
                priority
              />
            </motion.div>
          ) : (
            <div className="w-full h-full bg-secondary flex items-center justify-center">
              <span className="text-muted-foreground">No images available</span>
            </div>
          )}
        </AnimatePresence>
        <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/20 to-transparent" />
        <div className="absolute bottom-6 left-6 right-6">
          <motion.h3
            className="text-3xl font-extrabold text-white tracking-tight"
            initial={{ opacity: 0, y: 10 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.2 }}
          >
            {branch.name}
          </motion.h3>
        </div>
      </div>

      <div className="p-10 space-y-8 flex-1 flex flex-col bg-card">
        <div className="space-y-6 flex-1">
          <div className="flex gap-5 items-start">
            <div className="w-12 h-12 rounded-2xl bg-gold/10 flex items-center justify-center flex-shrink-0">
              <MapPin className="w-6 h-6 text-gold" />
            </div>
            <div className="space-y-1">
              <p className="text-foreground font-bold text-lg tracking-wide uppercase text-xs opacity-60">Location</p>
              <p className="text-lg text-muted-foreground leading-relaxed">
                {branch.address}<br />
                {branch.city}, {branch.postal_code}
              </p>
            </div>
          </div>

          <div className="flex gap-5 items-start">
            <div className="w-12 h-12 rounded-2xl bg-gold/10 flex items-center justify-center flex-shrink-0">
              <Phone className="w-6 h-6 text-gold" />
            </div>
            <div className="space-y-1">
              <p className="text-foreground font-bold text-lg tracking-wide uppercase text-xs opacity-60">Contact</p>
              <p className="text-lg text-muted-foreground">+91 {branch.phone}</p>
            </div>
          </div>
        </div>

        <button className="w-full bg-gold hover:bg-gold-dark text-black font-extrabold py-5 rounded-2xl transition-all duration-500 flex items-center justify-center gap-3 shadow-2xl shadow-gold/20 transform hover:-translate-y-2 group/btn">
          View Services
          <ArrowRight className="w-6 h-6 group-hover/btn:translate-x-2 transition-transform duration-300" />
        </button>
      </div>
    </motion.div>
  )
}

export default function Branches() {
  const { branches, loading } = useBranches()

  if (loading) return null

  return (
    <section id="branches" className="py-24 px-4 md:px-6 lg:px-8 bg-background relative overflow-hidden">
      <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-transparent via-gold/20 to-transparent" />

      <div className="max-w-7xl mx-auto">
        <motion.div
          className="text-center mb-16"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, ease: "easeOut" }}
        >
          <h2 className="text-4xl md:text-5xl font-extrabold text-foreground mb-4">
            Our <span className="text-gold">Premium</span> Locations
          </h2>
          <p className="text-muted-foreground max-w-2xl mx-auto text-lg">
            Experience luxury at its finest in our curated studios across the city.
          </p>
        </motion.div>

        <div className="grid md:grid-cols-3 gap-8 lg:gap-12">
          {branches?.map((branch, index) => (
            <BranchCard key={branch.id} branch={branch} index={index} />
          ))}
        </div>
      </div>
    </section>
  )
}
