"use client"

import { useState, useRef, useEffect } from "react"
import { ArrowRight, Volume2, VolumeX } from "lucide-react"
import { motion } from "framer-motion"
import BookingModal from "./booking-modal"

export default function Hero() {
  const [bookingOpen, setBookingOpen] = useState(false)
  const [isMuted, setIsMuted] = useState(true)
  const videoRef = useRef<HTMLVideoElement>(null)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (videoRef.current) {
          if (entry.isIntersecting) {
            videoRef.current.play()
          } else {
            videoRef.current.pause()
          }
        }
      },
      { threshold: 0.1 }
    )

    if (videoRef.current) observer.observe(videoRef.current)
    return () => observer.disconnect()
  }, [])

  return (
    <>
      <section id="hero" className="relative min-h-screen flex items-center justify-center pt-20 pb-12 px-4 sm:px-6 lg:px-8 overflow-hidden">
        <div className="absolute inset-0 -z-10">
          <video
            ref={videoRef}
            autoPlay
            loop
            muted={isMuted}
            playsInline
            preload="auto"
            className="absolute inset-0 w-full h-full object-cover [will-change:transform] [transform:translateZ(0)]"
            style={{ imageRendering: "auto" }}
          >
            <source src="/videos/IMG_6887.MP4" type="video/mp4" />
          </video>
          <div className="absolute inset-0 bg-black/50 [transform:translateZ(0)]"></div>
        </div>

        <div className="absolute bottom-10 right-10 z-20">
          <button
            onClick={() => setIsMuted(!isMuted)}
            className="p-3 bg-black/20 backdrop-blur-md rounded-full text-white hover:bg-black/40 transition-all border border-white/10"
            aria-label={isMuted ? "Unmute" : "Mute"}
          >
            {isMuted ? <VolumeX size={24} /> : <Volume2 size={24} />}
          </button>
        </div>

        <div className="max-w-5xl mx-auto text-center space-y-8 relative z-10">
          <motion.h1
            className="text-5xl md:text-8xl font-black text-white leading-[1.1] tracking-tighter drop-shadow-2xl"
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
          >
            Elevate Your <span className="text-gold drop-shadow-[0_0_15px_rgba(212,175,55,0.5)]">Nail Art</span> Experience
          </motion.h1>

          <motion.p
            className="text-xl md:text-2xl text-white/90 max-w-3xl mx-auto font-medium drop-shadow-lg"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, delay: 0.2, ease: "easeOut" }}
          >
            Premier nail art academy and salon offering professional manicure, pedicure, and advanced nail techniques in
            our luxurious boutique.
          </motion.p>

          <motion.div
            className="flex flex-col sm:flex-row gap-6 justify-center pt-10"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, delay: 0.4, ease: "easeOut" }}
          >
            <button
              onClick={() => setBookingOpen(true)}
              className="px-10 py-5 bg-gold text-black font-bold rounded-2xl hover:bg-gold-dark transition-all flex items-center justify-center gap-3 group shadow-2xl hover:shadow-gold/40 transform hover:-translate-y-1"
            >
              Book An Appointment
              <ArrowRight size={20} className="group-hover:translate-x-1 transition-transform" />
            </button>
            <a
              href="#services"
              className="px-10 py-5 border-2 border-white/50 text-white font-bold rounded-2xl hover:bg-white/10 hover:border-white transition-all shadow-xl backdrop-blur-md transform hover:-translate-y-1 flex items-center justify-center"
            >
              Explore Services
            </a>
          </motion.div>

          <motion.div
            className="grid grid-cols-3 gap-8 pt-16 border-t border-white/10"
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 1, delay: 0.6 }}
          >
            {[
              { label: "Happy Clients", value: "10,000+" },
              { label: "Expert Artists", value: "30+" },
              { label: "Years Experience", value: "4+" }
            ].map((stat, i) => (
              <div key={i} className="space-y-1">
                <p className="text-4xl font-black text-gold drop-shadow-lg">{stat.value}</p>
                <p className="text-xs uppercase tracking-widest font-bold text-white/70">{stat.label}</p>
              </div>
            ))}
          </motion.div>
        </div>
      </section>

      <BookingModal open={bookingOpen} onOpenChange={setBookingOpen} />
    </>
  )
}
