"use client"

import { useEffect, useState } from "react"
import { Sparkles, Palette, Users, Award, Star, Heart } from "lucide-react"
import { motion } from "framer-motion"
import { createClient } from "@/lib/supabase/client"

interface Category {
  id: string
  name: string
}

interface SubCategory {
  id: string
  name: string
  category_id: string
}

interface GroupedCategory extends Category {
  subcategories: string[]
}

const iconMap: Record<string, any> = {
  "Nail Art": Palette,
  "Manicure": Sparkles,
  "Pedicure": Heart,
  "Academy": Users,
  "Treatments": Award,
  "default": Star
}

export default function Services() {
  const [categories, setCategories] = useState<GroupedCategory[]>([])
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    const fetchData = async () => {
      const supabase = createClient()

      const { data: cats } = await supabase.from("categories").select("id, name")
      const { data: subs } = await supabase.from("sub_categories").select("id, name, category_id")

      if (cats && subs) {
        const grouped = cats.map(cat => ({
          ...cat,
          subcategories: subs
            .filter(sub => sub.category_id === cat.id)
            .map(sub => sub.name)
        }))
        setCategories(grouped)
      }
      setIsLoading(false)
    }

    fetchData()
  }, [])

  const getIcon = (name: string) => {
    for (const [key, icon] of Object.entries(iconMap)) {
      if (name.toLowerCase().includes(key.toLowerCase())) return icon
    }
    return iconMap.default
  }

  if (isLoading) return null

  return (
    <section id="services" className="py-20 px-4 sm:px-6 lg:px-8 bg-secondary">
      <div className="max-w-7xl mx-auto">
        <motion.div
          className="text-center space-y-4 mb-16"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, ease: "easeOut" }}
        >
          <h2 className="text-4xl md:text-5xl font-bold text-foreground">Our Services</h2>
          <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
            From classic nail care to advanced artistic designs, we offer comprehensive services
          </p>
        </motion.div>

        <div className="grid md:grid-cols-2 gap-8">
          {categories.map((category, index) => {
            const Icon = getIcon(category.name)
            return (
              <motion.div
                key={category.id}
                className="bg-card rounded-xl p-8 border border-border/50 hover:border-gold/30 transition-all duration-300 group shadow-sm hover:shadow-gold/5"
                initial={{ opacity: 0, y: 30 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ margin: "-50px" }}
                transition={{ duration: 0.6, delay: index * 0.1, ease: "easeOut" }}
              >
                <div className="w-14 h-14 bg-gradient-to-br from-gold/20 to-accent/20 rounded-lg flex items-center justify-center mb-6 group-hover:from-gold/30 group-hover:to-accent/30 transition">
                  <Icon size={28} className="text-gold" />
                </div>

                <h3 className="text-xl font-bold text-foreground mb-4">{category.name}</h3>

                <div className="space-y-3">
                  {category.subcategories.map((sub, i) => (
                    <div key={i} className="flex items-center gap-3 text-sm">
                      <div className="w-1.5 h-1.5 bg-gold rounded-full"></div>
                      <span className="text-foreground font-medium">{sub}</span>
                    </div>
                  ))}
                </div>
              </motion.div>
            )
          })}
        </div>
      </div>
    </section>
  )
}
