"use client"
import { useEffect, useState } from "react"
import { createClient } from "@/lib/supabase/client"
import { Star, ChevronLeft, ChevronRight } from "lucide-react"
import { motion, AnimatePresence } from "framer-motion"

interface Testimonial {
  id: string
  message: string
  rating: number
  user_name: string
  user_image: string | null
}

const DEFAULT_TESTIMONIALS: Testimonial[] = [
  {
    id: "1",
    message:
      "The nail art here is absolutely stunning! The staff is professional, attentive, and the atmosphere is so relaxing. Definitely my new favorite place.",
    rating: 5,
    user_name: "Priya Sharma",
    user_image: null,
  },
  {
    id: "2",
    message:
      "I visited the academy courses and learned so much. The instructors are knowledgeable and patient. The facility is world-class with beautiful interiors.",
    rating: 5,
    user_name: "Ananya Patel",
    user_image: null,
  },
  {
    id: "3",
    message:
      "Best pedicure experience I've had! The attention to detail is incredible, and the use of premium products shows. Will definitely be coming back!",
    rating: 5,
    user_name: "Neha Gupta",
    user_image: null,
  },
  {
    id: "4",
    message:
      "The personalized service here is exceptional. They take time to understand what you want and deliver beyond expectations. Highly recommended!",
    rating: 5,
    user_name: "Meera Desai",
    user_image: null,
  },
]

export default function Testimonials() {
  const [testimonials, setTestimonials] = useState<Testimonial[]>(DEFAULT_TESTIMONIALS)
  const [loading, setLoading] = useState(true)
  const [currentIndex, setCurrentIndex] = useState(0)
  const [isHovering, setIsHovering] = useState(false)

  useEffect(() => {
    const fetchTestimonials = async () => {
      try {
        const supabase = createClient()
        const { data, error } = await supabase
          .from("testimonials")
          .select("id, message, rating, user_name, user_image")
          .eq("is_approved", true)
          .order("created_at", { ascending: false })
          .limit(6)

        if (error) {
          setTestimonials(DEFAULT_TESTIMONIALS)
        } else if (data && data.length > 0) {
          const transformed = data.map((item: any) => ({
            id: item.id,
            message: item.message,
            rating: item.rating,
            user_name: item.user_name || "Anonymous",
            user_image: item.user_image || null,
          }))
          setTestimonials(transformed)
        } else {
          setTestimonials(DEFAULT_TESTIMONIALS)
        }
      } catch (error) {
        setTestimonials(DEFAULT_TESTIMONIALS)
      } finally {
        setLoading(false)
      }
    }
    fetchTestimonials()
  }, [])

  useEffect(() => {
    if (loading || isHovering) return

    const autoPlayTimer = setInterval(() => {
      setCurrentIndex((prev) => (prev + 1) % testimonials.length)
    }, 5000)

    return () => clearInterval(autoPlayTimer)
  }, [loading, isHovering, testimonials.length])

  const nextSlide = () => {
    setCurrentIndex((prev) => (prev + 1) % testimonials.length)
  }

  const prevSlide = () => {
    setCurrentIndex((prev) => (prev - 1 + testimonials.length) % testimonials.length)
  }

  const visibleTestimonials = [
    testimonials[currentIndex],
    testimonials[(currentIndex + 1) % testimonials.length],
    testimonials[(currentIndex + 2) % testimonials.length],
  ]

  if (loading) return null

  return (
    <section className="py-24 px-4 md:px-6 lg:px-8 bg-background dark:bg-background overflow-hidden">
      <div className="max-w-7xl mx-auto">
        <motion.div
          className="text-center mb-16"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, ease: "easeOut" }}
        >
          <h2 className="text-4xl md:text-5xl font-bold mb-4 text-foreground">
            <span className="text-gold">What Our Clients Say</span>
          </h2>
          <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
            Discover why our clients love United Nails Academy
          </p>
        </motion.div>

        <div className="relative" onMouseEnter={() => setIsHovering(true)} onMouseLeave={() => setIsHovering(false)}>
          <div className="grid md:grid-cols-3 gap-6 mb-8">
            <AnimatePresence mode="popLayout">
              {visibleTestimonials.map((testimonial, idx) => (
                <motion.div
                  key={`${testimonial.id}-${currentIndex}-${idx}`}
                  className="group h-full bg-card dark:bg-card border border-border rounded-2xl p-8 shadow-md hover:shadow-gold/5 transition-all duration-300 hover:-translate-y-2 flex flex-col"
                  initial={{ opacity: 0, x: 20, scale: 0.95 }}
                  animate={{ opacity: 1, x: 0, scale: 1 }}
                  exit={{ opacity: 0, x: -20, scale: 0.95 }}
                  transition={{ duration: 0.5, ease: "easeOut" }}
                >
                  <div className="flex gap-1 mb-6">
                    {Array.from({ length: testimonial.rating }).map((_, i) => (
                      <Star key={i} className="w-5 h-5 fill-gold text-gold" />
                    ))}
                  </div>

                  <p className="text-foreground mb-8 italic leading-relaxed text-lg">{'"' + testimonial.message + '"'}</p>

                  <div className="flex items-center gap-3 mt-auto pt-6 border-t border-border">
                    <div className="w-10 h-10 overflow-hidden rounded-full bg-gradient-to-br from-gold to-primary flex items-center justify-center">
                      {testimonial.user_image ? (
                        <img src={testimonial.user_image} alt={testimonial.user_name} className="w-full h-full object-cover" />
                      ) : (
                        <span className="text-white font-bold text-sm">{testimonial.user_name.charAt(0)}</span>
                      )}
                    </div>
                    <div>
                      <p className="font-semibold text-foreground text-sm">{testimonial.user_name}</p>
                      <p className="text-xs text-muted-foreground">Verified Client</p>
                    </div>
                  </div>
                </motion.div>
              ))}
            </AnimatePresence>
          </div>

          <div className="flex items-center justify-center gap-4">
            <button
              onClick={prevSlide}
              className="p-3 rounded-full bg-gold hover:bg-gold-dark transition-all duration-200 text-white shadow-lg shadow-gold/10 transform hover:scale-110"
              aria-label="Previous testimonials"
            >
              <ChevronLeft className="w-6 h-6" />
            </button>

            <div className="flex gap-2">
              {Array.from({ length: Math.ceil(testimonials.length / 3) }).map((_, idx) => (
                <button
                  key={idx}
                  onClick={() => setCurrentIndex(idx * 3)}
                  className={`w-3 h-3 rounded-full transition-all duration-500 ${idx === Math.floor(currentIndex / 3) ? "bg-gold w-8" : "bg-border hover:bg-muted"
                    }`}
                  aria-label={`Go to testimonial group ${idx + 1}`}
                />
              ))}
            </div>

            <button
              onClick={nextSlide}
              className="p-3 rounded-full bg-gold hover:bg-gold-dark transition-all duration-200 text-white shadow-lg shadow-gold/10 transform hover:scale-110"
              aria-label="Next testimonials"
            >
              <ChevronRight className="w-6 h-6" />
            </button>
          </div>

          <div className="absolute -top-8 -left-4 w-20 h-20 bg-gold/5 rounded-full blur-3xl pointer-events-none" />
          <div className="absolute -bottom-8 -right-4 w-24 h-24 bg-primary/5 rounded-full blur-3xl pointer-events-none" />
        </div>

        <motion.div
          className="grid grid-cols-3 gap-8 mt-20 pt-12 border-t border-border"
          initial={{ opacity: 0, y: 10 }}
          whileInView={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, ease: "easeOut" }}
        >
          <div className="text-center">
            <p className="text-3xl font-bold text-gold mb-2">10,000+</p>
            <p className="text-muted-foreground">Happy Clients</p>
          </div>
          <div className="text-center">
            <p className="text-3xl font-bold text-gold mb-2">5★</p>
            <p className="text-muted-foreground">Average Rating</p>
          </div>
          <div className="text-center">
            <p className="text-3xl font-bold text-gold mb-2">3</p>
            <p className="text-muted-foreground">Premium Locations</p>
          </div>
        </motion.div>
      </div>
    </section>
  )
}
